import io
from tokenize import generate_tokens, NL, NEWLINE, INDENT, DEDENT, untokenize

from codeable_detectors.match import Match


class DetectorContext(object):
    def __init__(self, text_or_matches, file_name=None):
        self.file_name = file_name
        self.texts_and_positions = []
        if isinstance(text_or_matches, Match):
            text_or_matches = [text_or_matches]
        if isinstance(text_or_matches, list):
            for match in text_or_matches:
                self.texts_and_positions.append([match.text, match.position])
                if self.file_name is None:
                    # take file name from first match, if none was specified
                    self.file_name = match.file_name
        else:
            self.texts_and_positions = [[text_or_matches, 0]]

    def _match_parser_pattern(self, pattern=None, match_extractor=None,
                              use_last_token_as_start=False):
        new_matches = []
        for text, prior_position in self.texts_and_positions:
            for tokens, start, end in pattern.scanString(text):
                if use_last_token_as_start:
                    # here tokens[-1] is expected to be the start position added by the parse action
                    start = tokens[-1]
                result_text = match_extractor(text, start, end)
                print("*** MATCH = " + result_text + "\n***")
                new_matches.append(Match(self.file_name, result_text, prior_position + start))
        return new_matches

    def matches_pattern(self, pattern):
        return self._match_parser_pattern(pattern=pattern,
                                          match_extractor=lambda text, match_start, match_end: text[
                                                                                               match_start:match_end])

    def matches_patterns(self, patterns):
        matches = []
        for pattern in patterns:
            pattern_matches = self._match_parser_pattern(pattern=pattern,
                                                         match_extractor=lambda text, match_start, match_end: text[
                                                                                                              match_start:match_end])
            # if one pattern does not match, matches_patterns fails
            if not pattern_matches:
                return []
            else:
                matches.extend(pattern_matches)
        # all matches by all patterns are returned, if matches_patterns succeeds
        return matches

    # basic indented block matcher, not considering special case like indents/dedents 
    # in round braces are ignored, as in Python
    def match_indented_block(self, initial_pattern):
        new_matches = []
        for text, prior_position in self.texts_and_positions:
            try:
                lines_with_indents = _preparse_indentations(text)
            except ValueError:
                # incorrectly indented block
                return []
            result = ""
            indent = 0
            initial_indent = None
            first_indent_found = False
            initial_pattern_found = False
            position = -99999  # dummy init

            for indent_dedent, line in lines_with_indents:
                print("### INDENT/DEDENT = " + str(indent_dedent) + "    LINE= " + line)
                indent = indent + indent_dedent
                if initial_indent is None:
                    initial_indent = indent

                if not initial_pattern_found:
                    if indent == initial_indent:
                        for tokens, start, end in initial_pattern.scanString(line):
                            print("INITIAL PATTERN!  " + str(indent))
                            initial_pattern_found = True
                            position = start
                            break
                else:
                    if not first_indent_found:
                        if indent_dedent == 1:
                            first_indent_found = True
                            result = result + line + "\n"
                        elif not (indent_dedent == 0 and line.strip() == ""):
                            # not a correctly indented block
                            return []
                    else:
                        if indent <= initial_indent:
                            break
                        else:
                            result = result + line + "\n"
            print("RESULT INDENTED BLOCK = " + result + "*********************************")
            new_matches.append(Match(self.file_name, result, prior_position + position))
        return new_matches

    def match_indented_python_block(self, initial_pattern):
        new_matches = []
        for text, prior_position in self.texts_and_positions:
            for tokens, start, end in initial_pattern.scanString(text):
                print("INDENTED BLOCK MATCHER, START AT: " + str(end))
                match = text[end:]
                # print("*** MATCH = " + match)
                position = start

                result = []
                indent = 1
                first_indent_found = False
                for token in generate_tokens(io.StringIO(match).readline):
                    token_type = token[0]
                    # token_start = token[2]
                    # token_end = token[3]
                    # token_content = token[4]
                    # print("token type: " + str(token_type) + " token = " + token_content)
                    if not first_indent_found:
                        if token_type not in [NL, NEWLINE, INDENT]:
                            # not a correctly indented block
                            return []
                        if token_type == INDENT:
                            first_indent_found = True
                    else:
                        if token_type == INDENT:
                            indent = indent + 1
                        if token_type == DEDENT:
                            indent = indent - 1
                    if indent <= 0:
                        # print("found end of indented block: " + str(token))
                        break
                    else:
                        result.append(token)

                print("RESULT INDENTED BLOCK = " + untokenize(result) + "*********************************")
                new_matches.append([untokenize(result), prior_position + position])
        return new_matches


def _guess_space_indentation(lines):
    for line in lines:
        leading_spaces = len(line) - len(line.lstrip(' '))
        if leading_spaces != 0:
            return leading_spaces
    # use indentation of 2 as a default if there are no leading spaces in all lines
    return 2


def _preparse_indentations(text):
    lines = text.splitlines()
    # guess the equivalent of space indentations to tabs by the first space indentation
    # in order to be able to expand tabs
    space_indentation = _guess_space_indentation(lines)
    lines_with_indents = []
    indentations = []
    indent_level = 0
    line_number = 0
    for line in lines:
        print("+++LINE= " + line)
        line_number = line_number + 1
        line_indents_dedents = 0
        no_tabs_line = line.expandtabs(space_indentation)
        leading_spaces = len(no_tabs_line) - len(no_tabs_line.lstrip(' '))
        indentations_leading_spaces = sum(indentations)
        print("+++LeadingSpaces = " + str(leading_spaces) + ", indents = " + str(indentations))
        if leading_spaces > indentations_leading_spaces:
            indentations.append(leading_spaces - indentations_leading_spaces)
            line_indents_dedents = 1
            indent_level = indent_level + 1
            print("+ INDENT")
        if leading_spaces < indentations_leading_spaces:
            while leading_spaces < indentations_leading_spaces:
                last_indentation = indentations[-1]
                indentations = indentations[:-1]
                indentations_leading_spaces = indentations_leading_spaces - last_indentation
                line_indents_dedents = line_indents_dedents - 1
                indent_level = indent_level - 1
                print("+ DEDENT")
            if leading_spaces != indentations_leading_spaces:
                print("INDENT LEVEL = " + str(indent_level))
                print("+++LeadingSpaces = " + str(leading_spaces))
                print("+++indentations_leading_spaces = " + str(indentations_leading_spaces))
                if indent_level != 0 and leading_spaces > indentations_leading_spaces:
                    # if condition above checks that we don't raise an exception if we go
                    # below the original indent, so that e.g. the following is permitted:
                    #     image: rabbitmq:3.7-management-alpine
                    #     ports:
                    #       - "5672"
                    #   payment:
                    #     build:
                    # we don't want to raise the unbalanced exception for "payment" in the example
                    raise ValueError("unbalanced indentations in line " + str(line_number) + ": '" + line + "'")
        lines_with_indents.append([line_indents_dedents, line])
    return lines_with_indents
